<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CampaignResource\Pages;
use App\Filament\Resources\CampaignResource\RelationManagers;
use App\Models\Campaign;
use App\Models\Contact;
use App\Models\WhatsappTemplate;
use App\Jobs\SendCampaignMessage;
use App\Settings\GeneralSettings; // استدعاء الإعدادات العامة
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Placeholder;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Illuminate\Support\Str;

class CampaignResource extends Resource
{
    protected static ?string $model = Campaign::class;

    protected static ?string $navigationIcon = 'heroicon-o-megaphone';
    protected static ?string $navigationLabel = 'الحملات الإعلانية';
    protected static ?string $modelLabel = 'حملة';
    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        // جلب إعدادات النظام (لمعرفة التوقيت الزمني مثلاً)
        $settings = app(GeneralSettings::class);

        return $form
            ->schema([
                // ---------------------------------------------------------
                // القسم الأول: الإعدادات الأساسية
                // ---------------------------------------------------------
                Section::make('تفاصيل الحملة')
                    ->description('إعدادات الاسم والحساب المرسل.')
                    ->schema([
                        Hidden::make('user_id')
                            ->default(fn () => auth()->id())
                            ->required(),

                        TextInput::make('name')
                            ->label('اسم الحملة')
                            ->required()
                            ->placeholder('مثلاً: عروض الجمعة البيضاء')
                            ->columnSpan(2),

                        Select::make('whatsapp_account_id')
                            ->label('الإرسال عبر الحساب')
                            ->relationship(
                                name: 'whatsappAccount',
                                titleAttribute: 'name',
                                modifyQueryUsing: fn (Builder $query) => $query->where('user_id', auth()->id())
                            )
                            ->required()
                            ->preload()
                            ->searchable()
                            ->live()
                            ->afterStateUpdated(fn (Set $set) => $set('template_name', null)),

                        // --- قائمة القوالب (تعتمد على الحساب المختار) ---
                        Select::make('template_name')
                            ->label('اختر القالب')
                            ->options(function (Get $get) {
                                $accountId = $get('whatsapp_account_id');
                                if (!$accountId) return [];
                                
                                // جلب القوالب المعتمدة (APPROVED) فقط
                                return WhatsappTemplate::where('whatsapp_account_id', $accountId)
                                    ->where('status', 'APPROVED')
                                    ->pluck('name', 'name');
                            })
                            ->required()
                            ->live()
                            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                // 1. تصفير المتغيرات القديمة عند تغيير القالب
                                $set('template_variables', []);

                                // 2. ضبط لغة القالب تلقائياً
                                $accountId = $get('whatsapp_account_id');
                                $template = WhatsappTemplate::where('whatsapp_account_id', $accountId)
                                    ->where('name', $state)
                                    ->first();
                                
                                if ($template) {
                                    $set('template_language', $template->language);
                                }
                            })
                            ->helperText('تظهر هنا القوالب التي تمت مزامنتها من Meta وحالتها Active.'),
                            
                        TextInput::make('template_language')
                            ->label('لغة القالب')
                            ->readOnly()
                            ->required(),

                    ])->columns(2),

                // ---------------------------------------------------------
                // القسم الثاني: ملء المتغيرات (Dynamic Variables)
                // ---------------------------------------------------------
                Section::make('تخصيص الرسالة (المتغيرات)')
                    ->description('القالب المختار يحتوي على متغيرات، يرجى ملء القيم أدناه.')
                    ->visible(fn (Get $get) => $get('template_name') !== null)
                    ->schema(function (Get $get) {
                        $templateName = $get('template_name');
                        $accountId = $get('whatsapp_account_id');

                        if (!$templateName || !$accountId) return [];

                        // 1. جلب بيانات القالب
                        $template = WhatsappTemplate::where('whatsapp_account_id', $accountId)
                            ->where('name', $templateName)
                            ->first();

                        if (!$template || empty($template->components)) return [];

                        // 2. البحث عن المكون الذي يحتوي على النص (BODY)
                        // ملاحظة: قد تكون components مخزنة كـ array أو collection حسب إعدادات الموديل
                        $components = is_array($template->components) ? $template->components : json_decode($template->components, true);
                        $bodyComponent = collect($components)->firstWhere('type', 'BODY');
                        
                        if (!$bodyComponent || !isset($bodyComponent['text'])) return [];

                        $text = $bodyComponent['text'];

                        // 3. استخراج الأرقام بين الأقواس {{1}}, {{2}}
                        preg_match_all('/{{([0-9]+)}}/', $text, $matches);
                        $variableNumbers = array_unique($matches[1]);
                        sort($variableNumbers);

                        // إذا لم يوجد متغيرات
                        if (empty($variableNumbers)) {
                            return [
                                Placeholder::make('preview')
                                    ->label('معاينة النص')
                                    ->content($text)
                            ];
                        }

                        // 4. إنشاء حقول الإدخال ديناميكياً
                        $fields = [];
                        
                        // عرض النص الأصلي للمساعدة
                        $fields[] = Placeholder::make('template_preview')
                                    ->label('نص القالب الأصلي')
                                    ->content($text)
                                    ->columnSpanFull();

                        foreach ($variableNumbers as $num) {
                            $fields[] = TextInput::make("template_variables.{$num}")
                                ->label("المتغير رقم {{$num}}")
                                ->placeholder("القيمة البديلة للمتغير {{$num}}")
                                ->required()
                                ->helperText("سيتم استبدال {{$num}} في النص بالقيمة التي تكتبها هنا.");
                        }

                        return $fields;
                    }),

                // ---------------------------------------------------------
                // القسم الثالث: الجدولة (مع استخدام الإعدادات العامة)
                // ---------------------------------------------------------
                Section::make('توقيت الإرسال')
                    ->schema([
                        DateTimePicker::make('scheduled_at')
                            ->label('تاريخ ووقت البدء')
                            ->minDate(now())
                            ->seconds(false)
                            ->visible(fn () => auth()->user()->hasFeature('has_schedule_campaigns') ?? true) // افتراضياً متاح إذا لم توجد الخاصية
                            // هنا نستخدم الإعدادات العامة لعرض التوقيت
                            ->helperText(function() use ($settings) {
                                $msg = auth()->user()->hasFeature('has_schedule_campaigns') 
                                    ? 'سيتم جدولة الحملة وتشغيلها تلقائياً.' 
                                    : 'ميزة الجدولة غير متاحة في باقتك الحالية.';
                                
                                return $msg . " (توقيت النظام: {$settings->timezone})";
                            }),
                    ])
                    ->collapsed(),
            ]);
    }

    public static function table(Table $table): Table
    {
        // استخدام الإعدادات العامة في الجدول
        $settings = app(GeneralSettings::class);

        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('اسم الحملة')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->description(fn (Campaign $record) => Str::limit($record->template_name, 20)),

                TextColumn::make('whatsappAccount.name')
                    ->label('من الحساب')
                    ->icon('heroicon-m-phone')
                    ->color('gray'),
                
                TextColumn::make('status')
                    ->label('الحالة')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'pending' => 'gray',
                        'scheduled' => 'info',
                        'processing' => 'warning',
                        'completed' => 'success',
                        'failed' => 'danger',
                        default => 'gray',
                    })
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'pending' => 'بانتظار البدء',
                        'scheduled' => 'مجدولة',
                        'processing' => 'جاري الإرسال',
                        'completed' => 'مكتملة',
                        'failed' => 'فشلت',
                        default => $state,
                    }),

                TextColumn::make('scheduled_at')
                    ->label('موعد الجدولة')
                    ->dateTime('d/m/Y h:i A')
                    ->sortable()
                    ->toggleable()
                    ->placeholder('إرسال فوري'),
                    
                TextColumn::make('success_count')
                    ->label('تم الإرسال')
                    ->icon('heroicon-m-check-circle')
                    ->color('success')
                    ->alignCenter(),

                TextColumn::make('failed_count')
                    ->label('فشل')
                    ->icon('heroicon-m-x-circle')
                    ->color('danger')
                    ->alignCenter(),
                
                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                // زر التشغيل اليدوي (Start Now)
                Action::make('start_campaign')
                    ->label('تشغيل الآن')
                    ->icon('heroicon-o-play')
                    ->color('success')
                    ->requiresConfirmation()
                    ->modalHeading('تأكيد بدء الحملة')
                    ->modalDescription('سيتم خصم الرصيد وبدء الإرسال فوراً لجميع جهات الاتصال المسجلة.')
                    ->visible(fn (Campaign $record) => in_array($record->status, ['pending', 'failed'])) // يظهر للمحاولات الفاشلة أيضاً
                    ->action(function (Campaign $record) {
                        
                        $user = auth()->user();
                        
                        // 1. جلب جهات الاتصال المرتبطة بالمستخدم
                        // ملاحظة: يمكن تحسين هذا لربط جهات اتصال بمجموعة معينة (Groups) مستقبلاً
                        $contacts = Contact::where('user_id', $record->user_id)->get();
                        
                        if ($contacts->isEmpty()) {
                            Notification::make()->title('خطأ: لا توجد جهات اتصال')->body('أضف جهات اتصال أولاً للقائمة.')->warning()->send();
                            return;
                        }

                        // 2. التحقق من الرصيد
                        if ($contacts->count() > $user->credits) {
                            Notification::make()
                                ->title('رصيدك غير كافٍ')
                                ->body("رصيدك الحالي: {$user->credits} | المطلوب: {$contacts->count()}")
                                ->danger()
                                ->persistent()
                                ->send();
                            return;
                        }

                        // 3. تحديث الحالة
                        $record->update([
                            'status' => 'processing',
                            'total_contacts' => $contacts->count()
                        ]);

                        Notification::make()->title('جاري بدء الحملة...')->body('سيتم الإرسال في الخلفية.')->success()->send();

                        // 4. توزيع المهام (Dispatch Jobs)
                        foreach ($contacts as $index => $contact) {
                            // نمرر الحملة وجهة الاتصال للـ Job
                            SendCampaignMessage::dispatch($record, $contact)
                                ->delay(now()->addSeconds($index * rand(2, 5))); // تأخير عشوائي بسيط لتجنب الحظر
                        }
                    }),

                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->emptyStateHeading('لا توجد حملات بعد')
            ->emptyStateDescription('أنشئ حملتك الأولى وابدأ بالتواصل مع عملائك.')
            ->emptyStateIcon('heroicon-o-megaphone');
    }

    public static function getEloquentQuery(): Builder
    {
        // كل مستخدم يرى حملاته فقط
        return parent::getEloquentQuery()->where('user_id', auth()->id())->latest();
    }

    public static function getRelations(): array
    {
        return [
            // يمكن إضافة علاقة السجلات (Logs) هنا لرؤية تفاصيل كل رسالة
             RelationManagers\LogsRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCampaigns::route('/'),
            'create' => Pages\CreateCampaign::route('/create'),
            'edit' => Pages\EditCampaign::route('/{record}/edit'),
        ];
    }
}