<?php

namespace App\Filament\Resources;

use App\Filament\Resources\RoleResource\Pages;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Table;

// 🚀 استدعاء موديل Spatie Role
use Spatie\Permission\Models\Role as SpatieRole; // تم تغيير اسمه لتجنب التعارض
use Illuminate\Database\Eloquent\Builder;

// استدعاء مكونات Filament
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ViewAction;


class RoleResource extends Resource
{
    // 🚀 ربط المورد بموديل Spatie
    protected static ?string $model = SpatieRole::class;

    protected static ?string $navigationIcon = 'heroicon-o-shield-check';
    protected static ?string $navigationGroup = 'الإدارة والتحكم'; // تجميعها في شريط جانبي
    protected static ?string $navigationLabel = 'الأدوار';
    protected static ?string $modelLabel = 'دور';

    
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->label('اسم الدور')
                    ->required()
                    ->unique(ignoreRecord: true) // لا يسمح بالتكرار
                    ->minLength(2)
                    ->maxLength(255),
                
                // 🚀 حقل لاختيار وتعيين الصلاحيات للدور
                Select::make('permissions')
                    ->label('الصلاحيات الممنوحة')
                    ->multiple()
                    ->relationship('permissions', 'name') // ربط بعلاقة الصلاحيات
                    ->preload(), // تحميل مسبق للكل
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')
                    ->label('ID')
                    ->sortable(),
                TextColumn::make('name')
                    ->label('الاسم')
                    ->searchable(),
                TextColumn::make('permissions_count')
                    ->label('عدد الصلاحيات')
                    ->counts('permissions') // عرض عدد الصلاحيات المرتبطة
                    ->badge(),
                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                ViewAction::make(),
                EditAction::make(),
                DeleteAction::make()
                    // حماية الأدوار الأساسية من الحذف
                    ->visible(fn (SpatieRole $record) => !in_array($record->name, ['Super Admin', 'Client'])),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    // 🚀 منع عرض الدور 'Super Admin' و 'Client' في الجدول للموظفين العاديين
    public static function getEloquentQuery(): Builder
    {
        if (auth()->user()->hasRole('Super Admin')) {
            return parent::getEloquentQuery();
        }
        
        return parent::getEloquentQuery()->whereNotIn('name', ['Super Admin']);
    }

    public static function getRelations(): array
    {
        return [
            // يمكننا إضافة Relation Manager لربط المستخدمين بالدور هنا لاحقاً
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListRoles::route('/'),
            'create' => Pages\CreateRole::route('/create'),
            'edit' => Pages\EditRole::route('/{record}/edit'),
        ];
    }
}