<?php

namespace App\Filament\Resources;

use App\Filament\Resources\WhatsAppAccountResource\Pages;
use App\Models\WhatsAppAccount;
use App\Models\WhatsappTemplate;
use App\Services\WhatsAppService;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Get; // تم إضافته لـ Get::call
use Filament\Resources\Resource;
use Filament\Tables\Table;

use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;

use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Notifications\Notification;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\FileUpload; // يجب استيرادها لـ FileUpload

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Arr; // لاستخدام Arr::get

class WhatsAppAccountResource extends Resource
{
    protected static ?string $model = WhatsAppAccount::class;

    protected static ?string $navigationIcon = 'heroicon-o-device-phone-mobile';
    protected static ?string $navigationGroup = 'حسابات الواتساب';
    protected static ?string $navigationLabel = 'حسابات الواتساب';
    protected static ?string $modelLabel = 'حساب واتساب';

    /* =======================
     |        FORM
     ======================= */
    public static function form(Form $form): Form
    {
        return $form->schema([
            Select::make('user_id')
                ->label('العميل')
                ->relationship('user', 'name')
                ->default(auth()->id())
                ->required()
                ->visible(auth()->user()->hasRole('Super Admin')),

            TextInput::make('name')
                ->label('اسم الحساب')
                ->required(),

            TextInput::make('phone_number_id')
                ->label('Phone Number ID')
                ->required(),

            TextInput::make('waba_id')
                ->label('WABA ID')
                ->required(),

            Textarea::make('access_token')
                ->label('Access Token')
                ->required()
                ->rows(4),
        ])->columns(2);
    }

    /* =======================
     |        TABLE
     ======================= */
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')->label('اسم الحساب')->searchable(),
                TextColumn::make('user.name')->label('العميل')
                    ->visible(auth()->user()->hasRole('Super Admin')),
                IconColumn::make('is_active')->label('الحالة')->boolean(),
            ])
            ->actions([

                /* =======================
                 |  مزامنة القوالب
                 ======================= */
                Action::make('sync_templates')
                    ->label('مزامنة القوالب')
                    ->icon('heroicon-o-arrow-path')
                    ->color('warning')
                    ->requiresConfirmation()
                    ->action(fn (WhatsAppAccount $record) =>
                        app(WhatsAppService::class)->syncTemplates(
                            $record->id,
                            $record->access_token,
                            $record->waba_id
                        )
                    ),

                /* =======================
                 |  إرسال تجريبي
                 ======================= */
                Action::make('send_test')
                    ->label('إرسال تجريبي')
                    ->icon('heroicon-o-paper-airplane')
                    ->color('success')

                    ->form([
                        TextInput::make('to_number')
                            ->label('رقم المستلم')
                            ->required(),

                        Select::make('template_name')
                            ->label('القالب')
                            ->options(fn (WhatsAppAccount $record) =>
                                WhatsappTemplate::where('whatsapp_account_id', $record->id)
                                    ->where('status', 'APPROVED')
                                    ->pluck('name', 'name')
                            )
                            ->reactive() // 🔑 لضمان تفعيل النموذج الديناميكي
                            ->required(),

                        // ✅ ديناميكي بدون أخطاء
                        Section::make('مكونات القالب')
                            ->schema(function (Get $get, WhatsAppAccount $record) {

                                $schema = [];

                                $templateName = $get('template_name');
                                if (!$templateName) {
                                    return [];
                                }

                                $template = WhatsappTemplate::where(
                                    'whatsapp_account_id',
                                    $record->id
                                )->where('name', $templateName)->first();

                                if (!$template || !is_array($template->components)) {
                                    return [];
                                }
                                
                                $templateComponents = $template->components;
                                
                                foreach ($templateComponents as $component) {

                                    // BODY
                                    if ($component['type'] === 'BODY'
                                        && preg_match_all('/\{\{(\d+)\}\}/', $component['text'] ?? '', $m)
                                    ) {
                                        foreach ($m[1] as $i) {
                                            $schema[] = TextInput::make("body_params.$i")
                                                ->label("متغير الرسالة {{ $i }}")
                                                ->required();
                                        }
                                    }

                                    // HEADER (Media/Text)
                                    if ($component['type'] === 'HEADER') {
                                        $format = $component['format'] ?? null;
                                        
                                        if ($format === 'IMAGE') {
                                            $schema[] = FileUpload::make('header_image')
                                                ->label('صورة الرأس (إلزامي)')
                                                ->required()
                                                ->image()
                                                ->directory('whatsapp-media-uploads')
                                                ->visibility('private');
                                        } elseif ($format === 'DOCUMENT') {
                                             $schema[] = FileUpload::make('header_document')
                                                ->label('مستند الرأس (إلزامي)')
                                                ->required()
                                                ->directory('whatsapp-media-uploads')
                                                ->visibility('private');
                                        } elseif ($format === 'TEXT' && preg_match_all('/\{\{(\d+)\}\}/', $component['text'] ?? '', $m)) {
                                            foreach ($m[1] as $i) {
                                                $schema[] = TextInput::make("header_params.$i")
                                                    ->label("متغير العنوان {{ $i }}")
                                                    ->required();
                                            }
                                        }
                                    }
                                }

                                return $schema;
                            })
                            ->visible(fn (Get $get) => filled($get('template_name'))),
                    ])

                    ->action(function (WhatsAppAccount $record, array $data) {
                        
                        // 🔑 جلب القالب بالاسم (يجب أن يكون موجوداً)
                        $template = WhatsappTemplate::where(
                            'whatsapp_account_id',
                            $record->id
                        )->where('name', $data['template_name'])->first();
                        
                        // 🔑 لتنظيف الملفات المؤقتة
                        $uploadedMediaPaths = [];

                        if (!$template) {
                            Notification::make()
                                ->title('خطأ')
                                ->body('القالب غير موجود')
                                ->danger()
                                ->send();
                            return;
                        }

                        $service = app(WhatsAppService::class);

                        try {
                             // 🔑 بناء المكونات وتحميل الملفات (إذا وجدت)
                            $componentsApi = $service->buildApiComponents(
                                $data, 
                                $template,
                                $record->phone_number_id, // 🔑 تم التصحيح هنا
                                $record->access_token // 🔑 تم التصحيح هنا
                            );
                            
                            // تجميع مسارات الملفات التي تم رفعها محليًا بواسطة FileUpload
                            if (!empty($data['header_image'])) {
                                $uploadedMediaPaths[] = $data['header_image'];
                            }
                            if (!empty($data['header_document'])) {
                                $uploadedMediaPaths[] = $data['header_document'];
                            }

                            $result = $service->sendTemplate(
                                $record->phone_number_id,
                                $record->access_token,
                                $data['to_number'],
                                $template->name,
                                'en_US',
                                $componentsApi,
                                $record->user
                            );
                            
                            if ($result === true || ($result['success'] ?? false)) {
                                Notification::make()
                                    ->title('تم إرسال الاختبار')
                                    ->body('تم الخصم من رصيدك بنجاح. رصيدك المتبقي: ' . $record->user->fresh()->credits)
                                    ->success()
                                    ->send();
                            } else {
                                $errorMessage = Arr::get($result, 'error.message', 'حدث خطأ غير معروف في واجهة API.');
                                $errorCode = Arr::get($result, 'error.code', 0);
                                
                                Log::error("WhatsApp Test Send Failed: Error Code [{$errorCode}] - Message: {$errorMessage}");
                                Notification::make()->title('فشل الإرسال')->body("{$errorMessage}")->danger()->send();
                            }
                            
                        } catch (\Exception $e) {
                            Log::error('Test Send Global Exception: ' . $e->getMessage());
                            Notification::make()->title('فشل داخلي غير متوقع')->body($e->getMessage())->danger()->send();
                        } finally {
                            // 🗑️ تنظيف الملفات المؤقتة
                            if (!empty($uploadedMediaPaths)) {
                                Storage::disk('local')->delete($uploadedMediaPaths);
                            }
                        }
                    }),

                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    /* =======================
     |     الصلاحيات
     ======================= */
    public static function canCreate(): bool
    {
        $user = auth()->user();

        return $user->hasRole('Super Admin')
            || $user->whatsappAccounts()->count()
                < $user->getMaxWhatsappAccountsLimit();
    }

    public static function getEloquentQuery(): Builder
    {
        return auth()->user()->hasRole('Super Admin')
            ? parent::getEloquentQuery()
            : parent::getEloquentQuery()->where('user_id', auth()->id());
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListWhatsAppAccounts::route('/'),
            'create' => Pages\CreateWhatsAppAccount::route('/create'),
            'edit'   => Pages\EditWhatsAppAccount::route('/{record}/edit'),
        ];
    }
}