<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CampaignLog;
use App\Models\Message; // موديل الشات
use App\Models\WhatsAppAccount;
use Illuminate\Support\Facades\Log;

class WhatsAppWebhookController extends Controller
{
    // رمز التحقق (يجب أن يطابق ما ستكتبه في إعدادات Meta)
    protected $verifyToken = 'my_secure_token_123';

    /**
     * 1. مرحلة التحقق (Verification) - GET Request
     * يتصل فيسبوك بهذا الرابط ليتأكد من ملكيتك للسيرفر
     */
    public function verify(Request $request)
    {
        $mode = $request->query('hub_mode');
        $token = $request->query('hub_verify_token');
        $challenge = $request->query('hub_challenge');

        if ($mode && $token) {
            if ($mode === 'subscribe' && $token === $this->verifyToken) {
                return response($challenge, 200);
            }
        }

        return response('Forbidden', 403);
    }

    /**
     * 2. مرحلة استقبال البيانات (Notification) - POST Request
     * هنا تصل كافة التحديثات والرسائل
     */
    public function handle(Request $request)
    {
        $body = $request->all();

        // التحقق أن الطلب قادم من واتساب
        if (isset($body['object']) && $body['object'] == 'whatsapp_business_account') {
            
            foreach ($body['entry'] as $entry) {
                foreach ($entry['changes'] as $change) {
                    $value = $change['value'];

                    // -----------------------------------------------------------
                    // الحالة أ: تحديث حالة رسالة سابقة (Status Update)
                    // مثال: delivered, read, failed
                    // -----------------------------------------------------------
                    if (isset($value['statuses'])) {
                        foreach ($value['statuses'] as $status) {
                            $this->updateMessageStatus($status);
                        }
                    }

                    // -----------------------------------------------------------
                    // الحالة ب: رسالة واردة جديدة من عميل (Incoming Message)
                    // -----------------------------------------------------------
                    if (isset($value['messages'])) {
                        foreach ($value['messages'] as $message) {
                            // نمرر الـ metadata لمعرفة أي حساب استقبل الرسالة
                            $this->handleIncomingMessage($value['metadata'], $message);
                        }
                    }
                }
            }

            return response('EVENT_RECEIVED', 200);
        }

        return response('Not Found', 404);
    }

    /**
     * دالة تحديث حالة الرسائل (للتقارير والشات)
     */
    protected function updateMessageStatus($statusData)
    {
        $wamid = $statusData['id']; // معرف الرسالة عند فيسبوك
        $newStatus = $statusData['status']; // sent, delivered, read, failed

        // 1. تحديث سجلات الحملات (Campaign Logs)
        // لكي يعرف المستخدم نسبة نجاح حملته
        $log = CampaignLog::where('message_id', $wamid)->first();
        if ($log) {
            $log->update([
                'status' => $newStatus,
                'error_reason' => isset($statusData['errors']) ? json_encode($statusData['errors']) : null
            ]);
        }

        // 2. تحديث سجل الشات (Messages Table)
        // لكي تظهر علامة الصحين الأزرق في الشات
        $chatMessage = Message::where('wa_id', $wamid)->first();
        if ($chatMessage) {
            $chatMessage->update([
                'status' => $newStatus,
                // إذا قرأها العميل (read)، نعتبرها مقروءة من طرفه
                'is_read' => ($newStatus === 'read') 
            ]);
        }
    }

    /**
     * دالة حفظ الرسائل الواردة (للشات)
     */
    protected function handleIncomingMessage($metadata, $messageData)
    {
        // 1. تحديد الحساب المستقبل للرسالة
        $phoneId = $metadata['phone_number_id'];
        $account = WhatsAppAccount::where('phone_number_id', $phoneId)->first();

        // إذا وصل إشعار لحساب غير مسجل في نظامنا، نتجاهله
        if (!$account) {
            return;
        }

        // 2. استخراج بيانات الرسالة
        $from = $messageData['from']; // رقم العميل
        $waId = $messageData['id'];   // معرف الرسالة (لمنع التكرار)
        $type = $messageData['type'];

        // استخراج النص حسب نوع الرسالة
        $body = '';
        if ($type == 'text') {
            $body = $messageData['text']['body'];
        } else {
            // إذا كانت صورة أو فيديو، نكتب نوعها حالياً
            // (يمكن تطوير هذا الجزء لاحقاً لتحميل الوسائط)
            $body = "[Media Type: $type]"; 
        }

        // 3. الحفظ في قاعدة البيانات
        // نستخدم firstOrCreate لمنع تكرار الرسالة إذا أعاد الويب هوك إرسالها بالخطأ
        Message::firstOrCreate(
            ['wa_id' => $waId], 
            [
                'whatsapp_account_id' => $account->id,
                'phone' => $from,
                'body' => $body,
                'type' => $type,
                'direction' => 'incoming', // وارد
                'status' => 'received',
                'is_read' => false, // لم يقرأها الموظف في لوحة التحكم بعد
            ]
        );
        
        Log::info("Message received from $from on account {$account->name}");
    }
}