<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\WhatsappTemplate;
use App\Models\User; 
use Illuminate\Support\Facades\Storage;

class WhatsAppService
{
    // إصدار واجهة Meta API
    protected $apiVersion = 'v18.0';

    /**
     * دالة مساعدة لخصم الرصيد
     * @param User $user
     * @return array|bool
     */
    protected function handleCreditDeduction(User $user)
    {
        if ($user->credits < 1) {
            return [
                'success' => false,
                'error' => ['message' => 'الرصيد غير كافٍ لإرسال الرسالة.', 'code' => 402]
            ];
        }
        try {
            $user->decrement('credits', 1);
            return true;
        } catch (\Exception $e) {
            Log::error('Credit Deduction Failure: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => ['message' => 'فشل داخلي في تحديث قاعدة البيانات.', 'code' => 500]
            ];
        }
    }

    /**
     * رفع الملفات إلى WhatsApp API والحصول على Media ID
     */
    public function uploadMedia($phoneId, $token, $filePath, $mimeType)
    {
        $url = "https://graph.facebook.com/{$this->apiVersion}/{$phoneId}/media";
        try {
            $disk = Storage::disk('public');
            if (!$disk->exists($filePath)) {
                Log::error("WhatsApp Media Upload Failed: File not found at path: {$filePath}");
                return false;
            }
            $fileContents = $disk->get($filePath);
            $fileName = basename($filePath);

            $response = Http::withToken($token)
                ->attach('file', $fileContents, $fileName, ['Content-Type' => $mimeType]) 
                ->post($url, [
                    'type' => $mimeType,
                    'messaging_product' => 'whatsapp',
                ]);
            
            if ($response->successful()) {
                return $response->json()['id'] ?? false;
            }
            Log::error('WhatsApp Media Upload Failed: ' . $response->body());
            return false;
        } catch (\Exception $e) {
            Log::error('WhatsApp Media Upload Exception: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * بناء هيكل المكونات (Components) لأي قالب WhatsApp
     */
    public function buildApiComponents(array $formData, WhatsappTemplate $template, $phoneId, $token)
    {
        $apiComponents = [];
        $templateComponents = $template->components;

        if (!is_array($templateComponents)) {
            $templateComponents = json_decode($template->components, true) ?? [];
        }

        $disk = Storage::disk('public');

        foreach ($templateComponents as $component) {
            switch ($component['type']) {
                case 'HEADER':
                    $headerComponent = ['type' => 'header', 'parameters' => []];
                    $format = $component['format'] ?? 'TEXT';

                    if ($format === 'TEXT' && !empty($formData['header_params'])) {
                        ksort($formData['header_params']);
                        foreach ($formData['header_params'] as $value) {
                            $headerComponent['parameters'][] = ['type' => 'text', 'text' => $value];
                        }
                    } elseif ($format === 'IMAGE' && !empty($formData['header_image'])) {
                        $filePath = $formData['header_image'];
                        if ($disk->exists($filePath)) {
                            $mediaId = $this->uploadMedia($phoneId, $token, $filePath, $disk->mimeType($filePath));
                            if ($mediaId) {
                                $headerComponent['parameters'][] = ['type' => 'image', 'image' => ['id' => $mediaId]];
                            }
                        }
                    } elseif ($format === 'DOCUMENT' && !empty($formData['header_document'])) {
                        $filePath = $formData['header_document'];
                        if ($disk->exists($filePath)) {
                            $mediaId = $this->uploadMedia($phoneId, $token, $filePath, $disk->mimeType($filePath));
                            if ($mediaId) {
                                $headerComponent['parameters'][] = [
                                    'type' => 'document',
                                    'document' => ['id' => $mediaId, 'filename' => basename($filePath)]
                                ];
                            }
                        }
                    }

                    if (!empty($headerComponent['parameters'])) {
                        $apiComponents[] = $headerComponent;
                    }
                    break;

                case 'BODY':
                    if (!empty($formData['body_params'])) {
                        $bodyComponent = ['type' => 'body', 'parameters' => []];
                        ksort($formData['body_params']);
                        foreach ($formData['body_params'] as $value) {
                            $bodyComponent['parameters'][] = ['type' => 'text', 'text' => $value];
                        }
                        $apiComponents[] = $bodyComponent;
                    }
                    break;

                case 'BUTTONS':
                    if (!empty($formData['buttons'])) {
                        $buttonComponent = ['type' => 'button', 'parameters' => []];
                        foreach ($formData['buttons'] as $btn) {
                            $buttonComponent['parameters'][] = [
                                'type' => $btn['type'], 
                                'text' => $btn['text'], 
                                'url'  => $btn['url'] ?? null
                            ];
                        }
                        $apiComponents[] = $buttonComponent;
                    }
                    break;

                case 'CAROUSEL':
                    if (!empty($component['cards'])) {
                        $carouselComponent = ['type' => 'carousel', 'cards' => []];
                        foreach ($component['cards'] as $card) {
                            $cardData = ['components' => []];
                            foreach ($card['components'] as $cardComp) {
                                if ($cardComp['type'] === 'HEADER' && !empty($formData['card_headers'])) {
                                    $filePath = $formData['card_headers'][$cardComp['id']] ?? null;
                                    if ($filePath && $disk->exists($filePath)) {
                                        $mediaId = $this->uploadMedia($phoneId, $token, $filePath, $disk->mimeType($filePath));
                                        $cardData['components'][] = ['type' => 'header', 'parameters' => [['type'=>'image','image'=>['id'=>$mediaId]]]];
                                    }
                                }

                                if ($cardComp['type'] === 'BODY' && !empty($formData['card_bodies'])) {
                                    $text = $formData['card_bodies'][$cardComp['id']] ?? $cardComp['text'] ?? '';
                                    $cardData['components'][] = ['type' => 'body', 'parameters' => [['type'=>'text','text'=>$text]]];
                                }

                                if ($cardComp['type'] === 'BUTTONS' && !empty($formData['card_buttons'])) {
                                    $buttons = [];
                                    foreach ($formData['card_buttons'][$cardComp['id']] ?? [] as $b) {
                                        $buttons[] = ['type'=>$b['type'],'text'=>$b['text'],'url'=>$b['url']??null];
                                    }
                                    $cardData['components'][] = ['type'=>'button','parameters'=>$buttons];
                                }
                            }
                            $carouselComponent['cards'][] = $cardData;
                        }
                        $apiComponents[] = $carouselComponent;
                    }
                    break;
            }
        }

        if (empty($apiComponents) && empty($formData)) {
            $apiComponents[] = ['type' => 'body', 'parameters' => [['type'=>'text','text'=>'لا توجد مدخلات مطلوبة للقالب النصي']]];
        }

        return $apiComponents;
    }

    /**
     * إرسال قالب (Template) للحملات
     */
    public function sendTemplate($phoneId, $token, $to, $templateName, $languageCode = 'en_US', array $components = [], User $user)
    {
        $deductionResult = $this->handleCreditDeduction($user);
        if ($deductionResult !== true) {
            return $deductionResult;
        }

        $url = "https://graph.facebook.com/{$this->apiVersion}/{$phoneId}/messages";

        $payload = [
            'messaging_product' => 'whatsapp',
            'to' => $to,
            'type' => 'template',
            'template' => [
                'name' => $templateName,
                'language' => ['code' => $languageCode],
                'components' => $components
            ]
        ];

        try {
            $response = Http::withToken($token)->post($url, $payload);
            if ($response->successful()) {
                return ['success' => true, 'data' => $response->json()];
            }

            $user->increment('credits', 1);
            Log::error('WhatsApp API Error (Send Template): ' . $response->body());
            return ['success' => false, 'error' => $response->json()];
        } catch (\Exception $e) {
            $user->increment('credits', 1);
            Log::error('WhatsApp Exception: ' . $e->getMessage());
            return ['success' => false, 'error' => ['message' => $e->getMessage(), 'code' => 500]];
        }
    }

    /**
     * إرسال نص عادي (Live Chat)
     */
    public function sendText($phoneId, $token, $to, $messageBody, User $user)
    {
        $deductionResult = $this->handleCreditDeduction($user);
        if ($deductionResult !== true) return false;

        $url = "https://graph.facebook.com/{$this->apiVersion}/{$phoneId}/messages";
        $payload = [
            'messaging_product' => 'whatsapp',
            'recipient_type' => 'individual',
            'to' => $to,
            'type' => 'text',
            'text' => ['preview_url' => false, 'body' => $messageBody]
        ];

        try {
            $response = Http::withToken($token)->post($url, $payload);
            if ($response->successful()) return true;

            $user->increment('credits', 1);
            return false;
        } catch (\Exception $e) {
            $user->increment('credits', 1);
            Log::error('WhatsApp Chat Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * مزامنة القوالب من Meta إلى قاعدة البيانات
     */
    public function syncTemplates($accountId, $token, $wabaId)
    {
        $url = "https://graph.facebook.com/{$this->apiVersion}/{$wabaId}/message_templates";

        try {
            $response = Http::withToken($token)->get($url, ['limit' => 100]);
            if ($response->successful()) {
                $templates = $response->json()['data'];
                foreach ($templates as $template) {
                    WhatsappTemplate::updateOrCreate(
                        ['whatsapp_account_id' => $accountId, 'name' => $template['name'], 'language' => $template['language']],
                        ['status' => $template['status'], 'category' => $template['category'], 'components' => $template['components']]
                    );
                }
                return true;
            }

            Log::error('WhatsApp Sync Error: ' . $response->body());
            return false;
        } catch (\Exception $e) {
            Log::error('WhatsApp Sync Exception: ' . $e->getMessage());
            return false;
        }
    }
}
